
from __future__ import annotations

from dataclasses import dataclass
from typing import Dict, List


@dataclass(frozen=True)
class FrameDefinition:
    name: str
    keywords: List[str]


FRAME_DEFINITIONS: List[FrameDefinition] = [
    FrameDefinition(
        name="ECONOMIC",
        keywords=[
            "tax", "taxes", "revenue", "spending", "deficit", "debt",
            "growth", "economy", "gdp", "investment", "markets",
            "business", "bills", "mortgage", "rates", "inflation",
        ],
    ),
    FrameDefinition(
        name="PUBLIC_SERVICES",
        keywords=[
            "nhs", "health", "hospital", "school", "education",
            "teacher", "welfare", "benefit", "council", "local government",
            "services", "care", "social care",
        ],
    ),
    FrameDefinition(
        name="INEQUALITY_FAIRNESS",
        keywords=[
            "inequality", "equal", "equity", "fair", "fairness",
            "poor", "low income", "rich", "wealthy", "poverty",
        ],
    ),
    FrameDefinition(
        name="POLITICAL_STRATEGY",
        keywords=[
            "labour", "conservative", "tory", "government", "opposition",
            "chancellor", "prime minister", "sunak", "starmer",
            "election", "vote", "polls", "politics", "strategy",
            "winners", "losers", "backbench", "parliament",
        ],
    ),
    FrameDefinition(
        name="CLIMATE_GREEN",
        keywords=[
            "climate", "green", "net zero", "environment",
            "carbon", "emissions", "energy transition",
        ],
    ),
    FrameDefinition(
        name="PUBLIC_REACTION",
        keywords=[
            "react", "reaction", "say", "warn", "anger",
            "praise", "critics", "backlash", "public", "voters",
            "business leaders", "unions", "teachers", "nurses",
        ],
    ),
]

FRAME_LABELS_ZH: Dict[str, str] = {
    "ECONOMIC": "经济框架（税收、赤字、增长、市场反应…）",
    "PUBLIC_SERVICES": "公共服务框架（NHS、学校、福利、地方政府…）",
    "INEQUALITY_FAIRNESS": "不平等/公平框架（贫富差距、公平性…）",
    "POLITICAL_STRATEGY": "政治博弈框架（政党竞争、选举、政治策略…）",
    "CLIMATE_GREEN": "气候/绿色框架（减排、绿色投资…）",
    "PUBLIC_REACTION": "公众/利益相关者反应框架（舆论、工会、行业…）",
    "OTHER": "其它/无法归类",
}


def classify_headline(headline: str) -> str:

    if not headline:
        return "OTHER"

    text = headline.lower()
    best_frame = "OTHER"
    best_score = 0

    for frame in FRAME_DEFINITIONS:
        score = 0
        for kw in frame.keywords:
            if kw in text:
                score += 1

        if score > best_score:
            best_score = score
            best_frame = frame.name

    return best_frame


def explain_frame(frame: str) -> str:

    return FRAME_LABELS_ZH.get(frame, frame)
